<?php

namespace Modules\KouffeSaas\Repositories;

use Illuminate\Support\Facades\Cache;
use Modules\KouffeSaas\Models\SaasSubscription;

class LicenseRepository
{
    protected $initRepository;

    public function __construct()
    {
        $this->initRepository = new InitRepository();
    }

    /**
     * Désactive un module (bloque l'accès sans supprimer les données)
     */
    public function disableModule(string $moduleName = 'kouffe-saas-addon'): void
    {
        // Marquer le module comme désactivé dans le cache
        Cache::put("module_disabled_{$moduleName}", true, 86400);
        
        // Log de l'événement
        \Log::warning("Module {$moduleName} has been disabled");
    }

    /**
     * Active un module
     */
    public function enableModule(string $moduleName = 'kouffe-saas-addon'): void
    {
        Cache::forget("module_disabled_{$moduleName}");
        \Log::info("Module {$moduleName} has been enabled");
    }

    /**
     * Vérifie si un module est désactivé
     */
    public function isModuleDisabled(string $moduleName = 'kouffe-saas-addon'): bool
    {
        return Cache::get("module_disabled_{$moduleName}", false);
    }

    /**
     * Vérifie la licence complète (module + abonnement)
     */
    public function checkLicense(int $schoolId): bool
    {
        // Vérifier la licence du module
        if (!$this->initRepository->check()) {
            return false;
        }

        // Vérifier si le module est désactivé
        if ($this->initRepository->isDisabled()) {
            return false;
        }

        // Vérifier l'abonnement de l'école
        if (!$this->initRepository->checkSubscription($schoolId)) {
            return false;
        }

        return true;
    }

    /**
     * Bloque l'accès si l'abonnement est expiré
     */
    public function blockAccessIfExpired(int $schoolId): bool
    {
        $subscription = SaasSubscription::where('school_id', $schoolId)
            ->latest()
            ->first();

        if (!$subscription || $subscription->isExpired()) {
            $this->disableModule();
            return false;
        }

        return true;
    }
}

