<?php

namespace Modules\KouffeSaas\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class SaasInvoice extends Model
{
    use SoftDeletes;

    protected $table = 'saas_invoices';

    protected $fillable = [
        'school_id',
        'subscription_id',
        'invoice_number',
        'amount',
        'currency',
        'status',
        'due_date',
        'paid_at',
        'payment_method',
        'payment_id',
        'notes',
        'items',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'due_date' => 'datetime',
        'paid_at' => 'datetime',
        'items' => 'array',
    ];

    /**
     * Relations
     */
    public function school()
    {
        return $this->belongsTo(\App\Models\SmSchool::class, 'school_id');
    }

    public function subscription()
    {
        return $this->belongsTo(SaasSubscription::class, 'subscription_id');
    }

    /**
     * Scopes
     */
    public function scopePaid($query)
    {
        return $query->where('status', 'paid');
    }

    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Generate unique invoice number
     */
    public static function generateInvoiceNumber(): string
    {
        $prefix = 'INV-' . date('Y');
        $lastInvoice = self::where('invoice_number', 'like', $prefix . '%')
            ->orderBy('id', 'desc')
            ->first();

        if ($lastInvoice) {
            $number = (int) substr($lastInvoice->invoice_number, -6) + 1;
        } else {
            $number = 1;
        }

        return $prefix . '-' . str_pad($number, 6, '0', STR_PAD_LEFT);
    }

    /**
     * Mark invoice as paid
     */
    public function markAsPaid(string $paymentMethod, string $paymentId): void
    {
        $this->status = 'paid';
        $this->paid_at = now();
        $this->payment_method = $paymentMethod;
        $this->payment_id = $paymentId;
        $this->save();
    }
}

