<?php

namespace Modules\KouffeSaas\Repositories;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Cache;
use Modules\KouffeSaas\Models\SaasSubscription;

class InitRepository
{
    protected $modulePath;
    protected $configFile;

    public function __construct()
    {
        $this->modulePath = module_path('KouffeSaas');
        $this->configFile = $this->modulePath . '/KouffeSaas.json';
    }

    /**
     * Vérifie la licence et l'état du module
     */
    public function check(): bool
    {
        // Vérifier si le fichier de config existe
        if (!File::exists($this->configFile)) {
            return false;
        }

        $config = json_decode(File::get($this->configFile), true);

        // Vérifier les paramètres essentiels
        if (empty($config['item_id']) || empty($config['checksum'])) {
            return false;
        }

        // Vérifier le domaine installé
        $installedDomain = $config['installed_domain'] ?? '';
        $currentDomain = request()->getHost();

        if (!empty($installedDomain) && $installedDomain !== $currentDomain) {
            return false;
        }

        // Vérifier la validité via l'API de licence
        return $this->validateLicense($config);
    }

    /**
     * Valide la licence via l'API auth.uxseven.com
     */
    protected function validateLicense(array $config): bool
    {
        $cacheKey = 'kouffe_saas_license_' . md5($config['item_id'] . $config['checksum']);

        return Cache::remember($cacheKey, 3600, function () use ($config) {
            try {
                $url = 'https://auth.uxseven.com/api/license/verify';
                
                $response = $this->curlIt($url, [
                    'item_id' => $config['item_id'],
                    'checksum' => $config['checksum'],
                    'domain' => request()->getHost(),
                    'module' => 'kouffe-saas-addon',
                ]);

                if ($response && isset($response['status']) && $response['status'] === 'valid') {
                    return true;
                }

                return false;
            } catch (\Exception $e) {
                \Log::error('Kouffe SaaS License Validation Error: ' . $e->getMessage());
                return false;
            }
        });
    }

    /**
     * Effectue une requête cURL vers l'API de licence
     * Méthode publique pour être utilisée par InstallRepository
     */
    public function curlIt(string $url, array $data): ?array
    {
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Accept: application/json',
            ],
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 30,
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode === 200 && $response) {
            return json_decode($response, true);
        }

        return null;
    }


    /**
     * Vérifie si l'école a un abonnement actif
     */
    public function checkSubscription(int $schoolId): bool
    {
        $subscription = SaasSubscription::where('school_id', $schoolId)
            ->where('status', 'active')
            ->where('end_date', '>', now())
            ->first();

        return $subscription !== null;
    }

    /**
     * Désactive le module si la licence est invalide
     */
    public function disableModule(): void
    {
        // Logique similaire à LicenseRepository::disableModule
        // Bloque l'accès sans supprimer les données
        Cache::put('kouffe_saas_disabled', true, 86400);
    }

    /**
     * Active le module
     */
    public function enableModule(): void
    {
        Cache::forget('kouffe_saas_disabled');
    }

    /**
     * Vérifie si le module est désactivé
     */
    public function isDisabled(): bool
    {
        return Cache::get('kouffe_saas_disabled', false);
    }
}

