<?php

namespace Modules\KouffeSaas\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\KouffeSaas\Models\SaasPlan;
use Modules\KouffeSaas\Models\SaasSubscription;
use Modules\KouffeSaas\Models\SaasInvoice;
use Modules\KouffeSaas\Services\SubscriptionService;
use App\SmSchool;

class SuperAdminController extends Controller
{
    protected $subscriptionService;

    public function __construct(SubscriptionService $subscriptionService)
    {
        $this->subscriptionService = $subscriptionService;
    }

    /**
     * Tableau de bord super admin
     */
    public function dashboard()
    {
        $stats = [
            'total_schools' => \App\SmSchool::count(),
            'active_subscriptions' => SaasSubscription::active()->count(),
            'expired_subscriptions' => SaasSubscription::expired()->count(),
            'total_revenue' => SaasInvoice::paid()->sum('amount'),
            'pending_invoices' => SaasInvoice::pending()->count(),
        ];

        return view('kouffesaas::super-admin.dashboard', compact('stats'));
    }

    /**
     * Liste des institutions
     */
    public function institutions()
    {
        $schools = \App\SmSchool::with(['subscriptions' => function ($query) {
            $query->latest()->first();
        }])->paginate(20);

        return view('kouffesaas::super-admin.institutions', compact('schools'));
    }

    /**
     * Liste des plans
     */
    public function plans()
    {
        $plans = SaasPlan::ordered()->get();
        return view('kouffesaas::super-admin.plans', compact('plans'));
    }

    /**
     * Créer/Modifier un plan
     */
    public function storePlan(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:saas_plans,slug,' . $request->id,
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'currency' => 'required|string|size:3',
            'billing_cycle' => 'required|in:monthly,yearly',
            'max_students' => 'required|integer|min:0',
            'max_storage_gb' => 'required|integer|min:0',
            'custom_domain' => 'boolean',
            'is_active' => 'boolean',
            'trial_days' => 'integer|min:0',
        ]);

        if ($request->has('id')) {
            $plan = SaasPlan::findOrFail($request->id);
            $plan->update($validated);
        } else {
            $plan = SaasPlan::create($validated);
        }

        return redirect()->route('kouffe-saas.admin.plans')
            ->with('success', 'Plan enregistré avec succès');
    }

    /**
     * Liste des factures
     */
    public function invoices()
    {
        $invoices = SaasInvoice::with(['school', 'subscription.plan'])
            ->latest()
            ->paginate(20);

        return view('kouffesaas::super-admin.invoices', compact('invoices'));
    }

    /**
     * Statistiques globales
     */
    public function statistics()
    {
        $stats = [
            'subscriptions_by_plan' => SaasSubscription::selectRaw('plan_id, count(*) as count')
                ->groupBy('plan_id')
                ->with('plan')
                ->get(),
            'revenue_by_month' => SaasInvoice::paid()
                ->selectRaw('DATE_FORMAT(paid_at, "%Y-%m") as month, SUM(amount) as total')
                ->groupBy('month')
                ->orderBy('month')
                ->get(),
        ];

        return view('kouffesaas::super-admin.statistics', compact('stats'));
    }
}

