<?php

namespace Modules\KouffeSaas\Middleware;

use Closure;
use Illuminate\Http\Request;
use Modules\KouffeSaas\Models\SaasCustomDomain;
use Modules\KouffeSaas\Repositories\InitRepository;
use Modules\KouffeSaas\Repositories\LicenseRepository;
use App\SmSchool;

class MapSchoolToRequest
{
    protected $initRepository;
    protected $licenseRepository;

    public function __construct()
    {
        $this->initRepository = new InitRepository();
        $this->licenseRepository = new LicenseRepository();
    }

    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        // Vérifier la licence du module
        if (!$this->initRepository->check()) {
            abort(403, 'Module non autorisé');
        }

        if ($this->initRepository->isDisabled()) {
            abort(403, 'Module désactivé');
        }

        // Détecter l'ID de l'école en fonction de l'URL
        $schoolId = $this->detectSchoolId($request);

        if (!$schoolId) {
            // Si pas d'école détectée, vérifier si c'est une route super admin
            if (!$this->isSuperAdminRoute($request)) {
                abort(404, 'École non trouvée');
            }
        } else {
            // Vérifier l'abonnement de l'école
            if (!$this->licenseRepository->checkLicense($schoolId)) {
                abort(403, 'Abonnement expiré ou invalide');
            }

            // Ajouter l'ID de l'école à la requête
            $request->merge(['school_id' => $schoolId]);
            $request->attributes->set('school_id', $schoolId);
            
            // Définir le contexte de l'école pour l'application
            app()->instance('school_id', $schoolId);
        }

        return $next($request);
    }

    /**
     * Détecte l'ID de l'école en fonction de l'URL
     */
    protected function detectSchoolId(Request $request): ?int
    {
        $host = $request->getHost();
        $subdomain = $this->extractSubdomain($host);

        // Méthode 1: Via sous-domaine (ex: school1.kouffe.com)
        if ($subdomain) {
            $school = \App\SmSchool::where('school_name', $subdomain)
                ->orWhere('slug', $subdomain)
                ->first();

            if ($school) {
                return $school->id;
            }
        }

        // Méthode 2: Via domaine personnalisé
        $customDomain = SaasCustomDomain::where('domain', $host)
            ->where('is_active', true)
            ->where('is_verified', true)
            ->first();

        if ($customDomain) {
            return $customDomain->school_id;
        }

        // Méthode 3: Via paramètre de requête (pour les API)
        if ($request->has('school_id')) {
            return (int) $request->get('school_id');
        }

        // Méthode 4: Via header HTTP
        if ($request->hasHeader('X-School-Id')) {
            return (int) $request->header('X-School-Id');
        }

        return null;
    }

    /**
     * Extrait le sous-domaine de l'URL
     */
    protected function extractSubdomain(string $host): ?string
    {
        $parts = explode('.', $host);
        
        // Si on a au moins 3 parties (subdomain.domain.tld)
        if (count($parts) >= 3) {
            return $parts[0];
        }

        return null;
    }

    /**
     * Vérifie si c'est une route super admin
     */
    protected function isSuperAdminRoute(Request $request): bool
    {
        $superAdminRoutes = [
            'admin/kouffe-saas',
            'api/kouffe-saas/admin',
        ];

        $path = $request->path();

        foreach ($superAdminRoutes as $route) {
            if (str_starts_with($path, $route)) {
                return true;
            }
        }

        return false;
    }
}

