<?php

namespace Modules\Lesson\Http\Controllers;

use Exception;
use DataTables;
use App\SmClass;
use App\SmStaff;
use App\SmSection;
use App\SmSubject;
use App\YearCheck;
use App\SmAssignSubject;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\App;
use Brian2694\Toastr\Facades\Toastr;
use Illuminate\Support\Facades\Auth;
use Modules\Lesson\Entities\SmLesson;
use Illuminate\Support\Facades\Config;
use Modules\Lesson\Entities\LessonPlanner;
use Modules\Lesson\Entities\SmLessonTopic;
use Modules\Lesson\Entities\SmLessonTopicDetail;
use Modules\University\Repositories\Interfaces\UnCommonRepositoryInterface;

class SmTopicController extends Controller
{
    

    public function index()
    {
        try {
            $data = $this->loadTopic();
            if (moduleStatusCheck('University')) {
                return view('university::topic.topic', $data);
            }

            return view('lesson::topic.topic', $data);

        } catch (Exception $exception) {
            Toastr::error('Operation Failed', 'Failed');

            return redirect()->back();
        }
    }

    public function store(Request $request)
    {
        if (moduleStatusCheck('University')) {
            $request->validate(
                [
                    'un_session_id' => 'required',
                    'un_faculty_id' => 'sometimes|nullable',
                    'un_department_id' => 'required',
                    'un_academic_id' => 'required',
                    'un_semester_id' => 'required',
                    'un_semester_label_id' => 'required',
                    'un_subject_id' => 'required',
                    'lesson' => 'required',
                ],
            );
        } else {
            $request->validate(
                [
                    'class' => 'required',
                    'subject' => 'required',
                    'section' => 'required',
                    'lesson' => 'required',
                ],
            );
        }

        DB::beginTransaction();
        if (moduleStatusCheck('University')) {
            $is_duplicate = SmLessonTopic::where('school_id', Auth::user()->school_id)
                ->where('un_session_id', $request->un_session_id)
                ->when($request->un_faculty_id, function ($query) use ($request): void {
                    $query->where('un_faculty_id', $request->un_faculty_id);
                })->where('un_department_id', $request->un_department_id)
                ->where('un_academic_id', $request->un_academic_id)
                ->where('un_semester_id', $request->un_department_id)
                ->where('un_semester_label_id', $request->un_academic_id)
                ->where('un_subject_id', $request->un_subject_id)
                ->where('lesson_id', $request->lesson)
                ->first();
        } else {
            $is_duplicate = SmLessonTopic::where('school_id', Auth::user()->school_id)
                            ->where('class_id', $request->class)
                            ->where('lesson_id', $request->lesson)
                            ->where('section_id', $request->section)
                            ->when(shiftEnable(), function ($query) use ($request) {
                                return $query->where('shift_id', $request->shift);
                            })
                            ->where('subject_id', $request->subject)
                            ->where('academic_id', getAcademicId())
                            ->first();
        }

        if ($is_duplicate) {
            $length = count($request->topic);
            for ($i = 0; $i < $length; $i++) {
                $topicDetail = new SmLessonTopicDetail;
                $topic_title = $request->topic[$i];
                $topicDetail->topic_id = $is_duplicate->id;
                $topicDetail->topic_title = $topic_title;
                $topicDetail->lesson_id = $request->lesson;
                $topicDetail->school_id = Auth::user()->school_id;
                if (moduleStatusCheck('University')) {
                    $topicDetail->un_academic_id = getAcademicId();
                } else {
                    $topicDetail->academic_id = getAcademicId();
                }

                $topicDetail->save();
            }

            DB::commit();
            Toastr::success('Operation successful', 'Success');

            return redirect()->back();
        }

        try {
            $smLessonTopic = new SmLessonTopic;
            $smLessonTopic->class_id = $request->class;
            $smLessonTopic->section_id = $request->section;
            $smLessonTopic->shift_id = shiftEnable() ? $request->shift : null;
            $smLessonTopic->subject_id = $request->subject;
            $smLessonTopic->lesson_id = $request->lesson;
            $smLessonTopic->created_at = YearCheck::getYear().'-'.date('m-d h:i:s');
            $smLessonTopic->school_id = Auth::user()->school_id;
            if (moduleStatusCheck('University')) {
                $common = App::make(UnCommonRepositoryInterface::class);
                $common->storeUniversityData($smLessonTopic, $request);
            } else {
                $smLessonTopic->academic_id = getAcademicId();
            }

            $smLessonTopic->save();
            $smTopic_id = $smLessonTopic->id;
            $length = count($request->topic);
            for ($i = 0; $i < $length; $i++) {
                $topicDetail = new SmLessonTopicDetail;
                $topic_title = $request->topic[$i];
                $topicDetail->topic_id = $smTopic_id;
                $topicDetail->topic_title = $topic_title;
                $topicDetail->lesson_id = $request->lesson;
                $topicDetail->school_id = Auth::user()->school_id;
                if (! moduleStatusCheck('University')) {
                    $topicDetail->academic_id = getAcademicId();
                }

                $topicDetail->save();
            }

            DB::commit();
            Toastr::success('Operation successful', 'Success');
            return redirect()->back();
        } catch (Exception $exception) {
            Toastr::error('Operation Failed', 'Failed');
            return redirect()->back();
        }
    }

    public function edit($id)
    {
        try {
            $data = $this->loadTopic();
            $data['topic'] = SmLessonTopic::where('academic_id', getAcademicId())
                ->where('id', $id)->where('school_id', Auth::user()->school_id)->first();
            $data['lessons'] = SmLesson::where('academic_id', getAcademicId())->where('school_id', Auth::user()->school_id)->get();
            $data['topicDetails'] = SmLessonTopicDetail::where('topic_id', $data['topic']->id)->where('academic_id', getAcademicId())
                ->where('school_id', Auth::user()->school_id)->get();
            if (shiftEnable()) {
                $data['shift_id'] = $data['topic']->shift_id;
            }
            $data['class_id'] = $data['topic']->class_id;
            $data['section_id'] = $data['topic']->section_id;
            if (moduleStatusCheck('University')) {

                $request = [
                    'semester_id' => $data['topic']->un_semester_id,
                    'academic_id' => $data['topic']->un_academic_id,
                    'session_id' => $data['topic']->un_session_id,
                    'department_id' => $data['topic']->un_department_id,
                    'faculty_id' => $data['topic']->un_faculty_id,
                    'semester_label_id' => $data['topic']->un_semester_label_id,
                    'subject_id' => $data['topic']->un_subject_id,
                ];
                $interface = App::make(UnCommonRepositoryInterface::class);
                $data += $interface->getCommonData($data['topic']);
                return view('university::topic.edit_topic', $data);
            }
            return view('lesson::topic.editTopic', $data);
        } catch (Exception $exception) {
            Toastr::error('Operation Failed', 'Failed');
            return redirect()->back();
        }
    }

    public function updateTopic(Request $request)
    {

        try {
            $length = count($request->topic);
            for ($i = 0; $i < $length; $i++) {
                $topicDetail = SmLessonTopicDetail::find($request->topic_detail_id[$i]);
                $topic_title = $request->topic[$i];
                $topicDetail->topic_title = $topic_title;
                $topicDetail->school_id = Auth::user()->school_id;
                $topicDetail->academic_id = getAcademicId();
                $topicDetail->save();
            }
            Toastr::success('Operation successful', 'Success');
            return redirect('/lesson/topic');
        } catch (Exception $exception) {
            Toastr::error('Operation Failed', 'Failed');
            return redirect()->back();
        }

    }

    public function topicdelete(Request $request)
    {
        $id = $request->id;
        $topic = SmLessonTopic::find($id);
        $topic->delete();
        $topicDetail = SmLessonTopicDetail::where('topic_id', $id)->get();
        if ($topicDetail) {
            foreach ($topicDetail as $data) {
                SmLessonTopicDetail::destroy($data->id);
                LessonPlanner::where('topic_detail_id', $data->id)->get();
            }
        }
        $topicLessonPlan = LessonPlanner::where('topic_id', $id)->get();
        if ($topicLessonPlan) {
            foreach ($topicLessonPlan as $topic_data) {
                LessonPlanner::destroy($topic_data->id);
            }
        }
        Toastr::success('Operation successful', 'Success');
        return redirect()->route('lesson.topic');
    }
    public function deleteTopicTitle($id)
    {
        SmLessonTopicDetail::destroy($id);
        $topicDetail = LessonPlanner::where('topic_detail_id', $id)->get();
        if ($topicDetail) {
            foreach ($topicDetail as $data) {
                LessonPlanner::destroy($data->id);
            }
        }

        Toastr::success('Operation successful', 'Success');

        return redirect()->back();
    }

    public function getAllTopicsAjax(Request $request)
    {
        //if ($request->ajax()) {
            if (Auth::user()->role_id === 4) {
                $subjects = SmAssignSubject::select('subject_id')
                    ->where('teacher_id', Auth()->user()->staff->id)
                    ->pluck('subject_id');

                $topics = SmLessonTopic::with(['lesson', 'class', 'section', 'subject', 'shift']) 
                    ->whereIn('subject_id', $subjects)
                    ->where('sm_lesson_topics.academic_id', getAcademicId())
                    ->where('sm_lesson_topics.school_id', Auth::user()->school_id);
            } else {
                $topics = SmLessonTopic::with(['lesson', 'class', 'section', 'subject', 'shift'])
                    ->where('sm_lesson_topics.academic_id', getAcademicId())
                    ->where('sm_lesson_topics.school_id', Auth::user()->school_id);
            }

            // Handle sorting
            if ($request->has('order')) {
                $orderColumnIndex = $request->input('order.0.column');
                $orderDirection = $request->input('order.0.dir');
                $orderColumn = $request->input('columns.'.$orderColumnIndex.'.name');

                switch ($orderColumn) {
                    case 'class_name':
                        $topics = $topics->join('sm_classes', 'sm_lesson_topics.class_id', '=', 'sm_classes.id')
                            ->orderBy('sm_classes.class_name', $orderDirection);
                        break;
                    case 'section_name':
                        $topics = $topics->join('sm_sections', 'sm_lesson_topics.section_id', '=', 'sm_sections.id')
                            ->orderBy('sm_sections.section_name', $orderDirection);
                        break;
                    case 'subject_name':
                        $topics = $topics->join('sm_subjects', 'sm_lesson_topics.subject_id', '=', 'sm_subjects.id')
                            ->orderBy('sm_subjects.subject_name', $orderDirection);
                        break;
                    case 'lesson_title':
                        $topics = $topics->join('sm_lessons', 'sm_lesson_topics.lesson_id', '=', 'sm_lessons.id')
                            ->orderBy('sm_lessons.lesson_title', $orderDirection);
                        break;
                    default:
                        $topics = $topics->orderBy($orderColumn, $orderDirection);
                        break;
                }
            }
            
            return DataTables::of($topics)
                ->addIndexColumn()
                ->addColumn('topics_name', function ($row): string {
                    $topics_name = '';
                    $topics_title = $row->topics;
                    foreach ($topics_title as $topic_title) {
                        $topics_name .= $topic_title->topic_title;
                        if (($topics_title->last()) !== $topic_title) {
                            $topics_name .= ',<br>';
                        }
                    }

                    return $topics_name;
                })
                ->addColumn('action', function ($row): string {
                    return '<div class="dropdown CRM_dropdown">
                        <button type="button" class="btn btn-secondary dropdown-toggle" data-toggle="dropdown">'.app('translator')->get('common.select').'</button>
                        <div class="dropdown-menu dropdown-menu-right">'.
                        (userPermission('topic-edit') ? '<a class="dropdown-item" href="'.route('topic-edit', $row->id).'">'.app('translator')->get('common.edit').'</a>' : '').
                        (userPermission('topic-delete') ? (Config::get('app.app_sync') ? '<span  data-toggle="tooltip" title="Disabled For Demo "><a  class="dropdown-item" href="#"  >'.app('translator')->get('common.disable').'</a></span>' :
                            '<a onclick="deleteTopic('.$row->id.');"  class="dropdown-item" href="#" data-id="'.$row->id.'"  >'.app('translator')->get('common.delete').'</a>') : '').
                        '</div>
                    </div>';
                })
                ->rawColumns(['action', 'topics_name'])
                ->make(true);
        //}

        return null;
    }

    public function loadTopic()
    {
        $auth = Auth::user();
        $teacher_info = SmStaff::where('user_id', $auth->id)->first();
        if (Auth::user()->role_id === 4) {
            $subjects = SmAssignSubject::select('subject_id')->where('teacher_id', $teacher_info->id)->get();
            $data['topics'] = SmLessonTopic::with(['lesson', 'class', 'section', 'subject'])->whereIn('subject_id', $subjects)->where('academic_id', getAcademicId())->where('school_id', $auth->school_id)->get();

        } else {
            $data['topics'] = SmLessonTopic::with(['lesson', 'class', 'section', 'subject'])->where('academic_id', getAcademicId())->where('school_id', $auth->school_id)->get();
        }

        if (! teacherAccess()) {
            $data['classes'] = SmClass::where('active_status', 1)->where('academic_id', getAcademicId())->where('school_id', $auth->school_id)->get();
        } else {
            $data['classes'] = SmAssignSubject::where('teacher_id', $teacher_info->id)
                ->join('sm_classes', 'sm_classes.id', 'sm_assign_subjects.class_id')
                ->where('sm_assign_subjects.active_status', 1)
                ->where('sm_assign_subjects.school_id', $auth->school_id)
                ->where('sm_assign_subjects.academic_id', getAcademicId())
                ->select('sm_classes.id', 'class_name')
                ->groupBy('sm_classes.id')
                ->get();
        }

        $data['subjects'] = SmSubject::select(['id', 'subject_name'])->get();
        $data['sections'] = SmSection::select(['id', 'section_name'])->get();

        return $data;
    }
    
}
