<?php

namespace Modules\KouffeSaas\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Carbon\Carbon;

class SaasSubscription extends Model
{
    use SoftDeletes;

    protected $table = 'saas_subscriptions';

    protected $fillable = [
        'school_id',
        'plan_id',
        'status',
        'start_date',
        'end_date',
        'trial_ends_at',
        'payment_method',
        'payment_id',
        'auto_renew',
        'cancelled_at',
        'cancellation_reason',
    ];

    protected $casts = [
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'trial_ends_at' => 'datetime',
        'auto_renew' => 'boolean',
        'cancelled_at' => 'datetime',
    ];

    /**
     * Relations
     */
    public function school()
    {
        return $this->belongsTo(\App\Models\SmSchool::class, 'school_id');
    }

    public function plan()
    {
        return $this->belongsTo(SaasPlan::class, 'plan_id');
    }

    public function invoices()
    {
        return $this->hasMany(SaasInvoice::class, 'subscription_id');
    }

    /**
     * Scopes
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active')
            ->where('end_date', '>', now());
    }

    public function scopeExpired($query)
    {
        return $query->where('status', 'expired')
            ->orWhere(function ($q) {
                $q->where('status', 'active')
                  ->where('end_date', '<=', now());
            });
    }

    /**
     * Check if subscription is active
     */
    public function isActive(): bool
    {
        return $this->status === 'active' && $this->end_date->isFuture();
    }

    /**
     * Check if subscription is expired
     */
    public function isExpired(): bool
    {
        return $this->status === 'expired' || 
               ($this->status === 'active' && $this->end_date->isPast());
    }

    /**
     * Check if subscription is in trial
     */
    public function isTrial(): bool
    {
        return $this->trial_ends_at && $this->trial_ends_at->isFuture();
    }

    /**
     * Check if subscription can be renewed
     */
    public function canRenew(): bool
    {
        return $this->auto_renew && 
               ($this->isActive() || $this->isExpired()) &&
               !$this->cancelled_at;
    }

    /**
     * Days remaining until expiration
     */
    public function daysRemaining(): int
    {
        if ($this->isExpired()) {
            return 0;
        }
        return max(0, now()->diffInDays($this->end_date, false));
    }
}

