<?php

namespace Modules\KouffeSaas\Repositories;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Modules\KouffeSaas\Models\SaasPlan;

class InstallRepository
{
    protected $modulePath;
    protected $configFile;

    public function __construct()
    {
        $this->modulePath = module_path('KouffeSaas');
        $this->configFile = $this->modulePath . '/KouffeSaas.json';
    }

    /**
     * Installe le module
     */
    public function install(array $data): bool
    {
        try {
            DB::beginTransaction();

            // Mettre à jour le fichier de configuration
            $this->updateConfig($data);

            // Exécuter les migrations
            Artisan::call('migrate', [
                '--path' => 'modules/KouffeSaas/Database/Migrations',
                '--force' => true,
            ]);

            // Créer les plans par défaut
            $this->createDefaultPlans();

            // Activer le module
            $this->activateKouffeModule($data);

            DB::commit();
            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Kouffe SaaS Installation Error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Met à jour le fichier de configuration
     */
    protected function updateConfig(array $data): void
    {
        $config = [];
        
        if (File::exists($this->configFile)) {
            $config = json_decode(File::get($this->configFile), true);
        }

        $config['item_id'] = $data['item_id'] ?? '';
        $config['checksum'] = $data['checksum'] ?? '';
        $config['installed_domain'] = request()->getHost();

        File::put($this->configFile, json_encode($config, JSON_PRETTY_PRINT));
    }

    /**
     * Crée les plans par défaut
     */
    protected function createDefaultPlans(): void
    {
        $plans = [
            [
                'name' => 'Trial',
                'slug' => 'trial',
                'description' => 'Plan d\'essai gratuit',
                'price' => 0,
                'currency' => 'USD',
                'billing_cycle' => 'monthly',
                'max_students' => 50,
                'max_storage_gb' => 5,
                'custom_domain' => false,
                'is_active' => true,
                'trial_days' => 14,
                'features' => [
                    'Jusqu\'à 50 étudiants',
                    '5 GB de stockage',
                    'Support par email',
                ],
                'sort_order' => 1,
            ],
            [
                'name' => 'Basic',
                'slug' => 'basic',
                'description' => 'Plan de base pour petites écoles',
                'price' => 49.99,
                'currency' => 'USD',
                'billing_cycle' => 'monthly',
                'max_students' => 200,
                'max_storage_gb' => 50,
                'custom_domain' => false,
                'is_active' => true,
                'trial_days' => 0,
                'features' => [
                    'Jusqu\'à 200 étudiants',
                    '50 GB de stockage',
                    'Support prioritaire',
                    'Rapports de base',
                ],
                'sort_order' => 2,
            ],
            [
                'name' => 'Premium',
                'slug' => 'premium',
                'description' => 'Plan premium pour grandes institutions',
                'price' => 149.99,
                'currency' => 'USD',
                'billing_cycle' => 'monthly',
                'max_students' => 0, // Unlimited
                'max_storage_gb' => 0, // Unlimited
                'custom_domain' => true,
                'is_active' => true,
                'trial_days' => 0,
                'features' => [
                    'Étudiants illimités',
                    'Stockage illimité',
                    'Domaine personnalisé',
                    'Support 24/7',
                    'Rapports avancés',
                    'API access',
                ],
                'sort_order' => 3,
            ],
        ];

        foreach ($plans as $planData) {
            SaasPlan::firstOrCreate(
                ['slug' => $planData['slug']],
                $planData
            );
        }
    }

    /**
     * Active le module Kouffe SaaS
     */
    public function activateKouffeModule(array $data): bool
    {
        $initRepo = new InitRepository();
        
        // Valider la licence via curlIt
        $url = 'https://auth.uxseven.com/api/license/activate';
        
        $response = $initRepo->curlIt($url, [
            'item_id' => $data['item_id'] ?? '',
            'checksum' => $data['checksum'] ?? '',
            'domain' => request()->getHost(),
            'module' => 'kouffe-saas-addon',
            'version' => '1.0.0',
        ]);

        if ($response && isset($response['status']) && $response['status'] === 'activated') {
            $initRepo->enableModule();
            return true;
        }

        return false;
    }

    /**
     * Méthode statique pour activer le module depuis l'extérieur
     */
    public static function activate(array $data): bool
    {
        $installRepo = new self();
        return $installRepo->activateKouffeModule($data);
    }

    /**
     * Désinstalle le module
     */
    public function uninstall(): bool
    {
        try {
            // Ne pas supprimer les données, juste désactiver
            $initRepo = new InitRepository();
            $initRepo->disableModule();
            
            return true;
        } catch (\Exception $e) {
            \Log::error('Kouffe SaaS Uninstallation Error: ' . $e->getMessage());
            return false;
        }
    }
}

