<?php

namespace Modules\KouffeSaas\Services;

use Modules\KouffeSaas\Models\SaasInvoice;
use Modules\KouffeSaas\Models\SaasSubscription;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use Stripe\Exception\ApiErrorException;

class PaymentService
{
    protected $stripeSecretKey;

    public function __construct()
    {
        $this->stripeSecretKey = config('services.stripe.secret');
        Stripe::setApiKey($this->stripeSecretKey);
    }

    /**
     * Traite un paiement Stripe
     */
    public function processStripePayment(int $invoiceId, string $paymentMethodId): array
    {
        $invoice = SaasInvoice::findOrFail($invoiceId);

        try {
            $paymentIntent = PaymentIntent::create([
                'amount' => (int) ($invoice->amount * 100), // Convertir en centimes
                'currency' => strtolower($invoice->currency),
                'payment_method' => $paymentMethodId,
                'confirmation_method' => 'manual',
                'confirm' => true,
                'return_url' => route('kouffe-saas.payment.return'),
            ]);

            if ($paymentIntent->status === 'succeeded') {
                $invoice->markAsPaid('stripe', $paymentIntent->id);
                
                // Activer ou renouveler l'abonnement
                $this->activateSubscription($invoice->subscription_id);

                return [
                    'success' => true,
                    'message' => 'Paiement effectué avec succès',
                ];
            }

            return [
                'success' => false,
                'message' => 'Le paiement nécessite une action supplémentaire',
                'client_secret' => $paymentIntent->client_secret,
            ];
        } catch (ApiErrorException $e) {
            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Traite un paiement PayPal
     */
    public function processPayPalPayment(int $invoiceId, string $paymentId): array
    {
        $invoice = SaasInvoice::findOrFail($invoiceId);

        // Logique PayPal (à implémenter selon votre intégration PayPal)
        // Pour l'instant, on simule le succès
        $invoice->markAsPaid('paypal', $paymentId);
        $this->activateSubscription($invoice->subscription_id);

        return [
            'success' => true,
            'message' => 'Paiement PayPal effectué avec succès',
        ];
    }

    /**
     * Active ou renouvelle un abonnement après paiement
     */
    protected function activateSubscription(int $subscriptionId): void
    {
        $subscription = SaasSubscription::findOrFail($subscriptionId);
        
        if ($subscription->isExpired()) {
            $subscriptionService = new \Modules\KouffeSaas\Services\SubscriptionService();
            $subscriptionService->renewSubscription($subscription);
        } else {
            $subscription->update(['status' => 'active']);
        }
    }

    /**
     * Configure le renouvellement automatique via Stripe
     */
    public function setupAutoRenewal(int $subscriptionId, string $paymentMethodId): bool
    {
        $subscription = SaasSubscription::findOrFail($subscriptionId);

        try {
            // Créer un customer Stripe si nécessaire
            // Créer un subscription Stripe pour le renouvellement automatique
            // Lier le subscription_id local avec le subscription_id Stripe

            $subscription->update([
                'payment_method' => 'stripe',
                'payment_id' => $paymentMethodId,
                'auto_renew' => true,
            ]);

            return true;
        } catch (\Exception $e) {
            \Log::error('Auto-renewal setup error: ' . $e->getMessage());
            return false;
        }
    }
}

