<?php

namespace Modules\KouffeSaas\Services;

use Modules\KouffeSaas\Models\SaasSubscription;
use Modules\KouffeSaas\Models\SaasPlan;
use Modules\KouffeSaas\Models\SaasInvoice;
use Carbon\Carbon;

class SubscriptionService
{
    /**
     * Crée un nouvel abonnement
     */
    public function createSubscription(int $schoolId, int $planId, bool $isTrial = false): SaasSubscription
    {
        $plan = SaasPlan::findOrFail($planId);
        
        $startDate = now();
        $endDate = $isTrial && $plan->trial_days > 0
            ? $startDate->copy()->addDays($plan->trial_days)
            : $startDate->copy()->addMonth();

        $trialEndsAt = $isTrial && $plan->trial_days > 0
            ? $endDate
            : null;

        $subscription = SaasSubscription::create([
            'school_id' => $schoolId,
            'plan_id' => $planId,
            'status' => 'active',
            'start_date' => $startDate,
            'end_date' => $endDate,
            'trial_ends_at' => $trialEndsAt,
            'auto_renew' => true,
        ]);

        return $subscription;
    }

    /**
     * Vérifie quotidiennement si la date actuelle > subscription_end_date
     */
    public function checkSubscription(): void
    {
        $expiredSubscriptions = SaasSubscription::where('status', 'active')
            ->where('end_date', '<=', now())
            ->get();

        foreach ($expiredSubscriptions as $subscription) {
            if ($subscription->auto_renew) {
                // Tentative de renouvellement automatique
                $this->renewSubscription($subscription);
            } else {
                // Marquer comme expiré
                $subscription->update(['status' => 'expired']);
            }
        }
    }

    /**
     * Renouvelle un abonnement
     */
    public function renewSubscription(SaasSubscription $subscription): bool
    {
        try {
            $plan = $subscription->plan;
            $newEndDate = $subscription->end_date->copy()->addMonth();

            $subscription->update([
                'end_date' => $newEndDate,
                'status' => 'active',
            ]);

            // Créer une facture pour le renouvellement
            $this->createInvoice($subscription, $plan->price);

            return true;
        } catch (\Exception $e) {
            \Log::error('Subscription Renewal Error: ' . $e->getMessage());
            $subscription->update(['status' => 'expired']);
            return false;
        }
    }

    /**
     * Annule un abonnement
     */
    public function cancelSubscription(int $subscriptionId, string $reason = null): bool
    {
        $subscription = SaasSubscription::findOrFail($subscriptionId);
        
        $subscription->update([
            'status' => 'cancelled',
            'auto_renew' => false,
            'cancelled_at' => now(),
            'cancellation_reason' => $reason,
        ]);

        return true;
    }

    /**
     * Crée une facture
     */
    public function createInvoice(SaasSubscription $subscription, float $amount): SaasInvoice
    {
        return SaasInvoice::create([
            'school_id' => $subscription->school_id,
            'subscription_id' => $subscription->id,
            'invoice_number' => SaasInvoice::generateInvoiceNumber(),
            'amount' => $amount,
            'currency' => $subscription->plan->currency,
            'status' => 'pending',
            'due_date' => now()->addDays(7),
            'items' => [
                [
                    'description' => $subscription->plan->name . ' - ' . $subscription->plan->billing_cycle,
                    'amount' => $amount,
                ],
            ],
        ]);
    }

    /**
     * Vérifie et applique les limites du plan
     */
    public function enforceLimits(int $schoolId, string $resourceType, int $currentCount): bool
    {
        $subscription = SaasSubscription::where('school_id', $schoolId)
            ->active()
            ->first();

        if (!$subscription) {
            return false;
        }

        $plan = $subscription->plan;

        switch ($resourceType) {
            case 'students':
                if (!$plan->isUnlimitedStudents() && $currentCount >= $plan->max_students) {
                    return false;
                }
                break;
            case 'storage':
                // La vérification du stockage nécessite une logique plus complexe
                // basée sur la taille réelle des fichiers
                break;
        }

        return true;
    }
}

