<?php

namespace Modules\KouffeSaas\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\KouffeSaas\Models\SaasSubscription;
use Modules\KouffeSaas\Models\SaasPlan;
use Modules\KouffeSaas\Models\SaasInvoice;
use Modules\KouffeSaas\Services\SubscriptionService;
use Modules\KouffeSaas\Services\DomainService;
use App\SmSchool;
use App\Student;

class SchoolAdminController extends Controller
{
    protected $subscriptionService;
    protected $domainService;

    public function __construct(
        SubscriptionService $subscriptionService,
        DomainService $domainService
    ) {
        $this->subscriptionService = $subscriptionService;
        $this->domainService = $domainService;
    }

    /**
     * Tableau de bord de l'école
     */
    public function dashboard(Request $request)
    {
        $schoolId = $request->attributes->get('school_id') ?? auth()->user()->school_id;
        
        $subscription = SaasSubscription::where('school_id', $schoolId)
            ->with('plan')
            ->latest()
            ->first();

        $invoices = SaasInvoice::where('school_id', $schoolId)
            ->latest()
            ->limit(5)
            ->get();

        $usage = [
            'students' => App\Student::where('school_id', $schoolId)->count(),
            'storage_gb' => $this->calculateStorageUsage($schoolId),
        ];

        return view('kouffesaas::school-admin.dashboard', compact(
            'subscription',
            'invoices',
            'usage'
        ));
    }

    /**
     * Affiche les plans disponibles
     */
    public function plans()
    {
        $plans = SaasPlan::active()->ordered()->get();
        $currentSubscription = SaasSubscription::where('school_id', auth()->user()->school_id)
            ->with('plan')
            ->latest()
            ->first();

        return view('kouffesaas::school-admin.plans', compact('plans', 'currentSubscription'));
    }

    /**
     * Souscrire à un plan
     */
    public function subscribe(Request $request)
    {
        $request->validate([
            'plan_id' => 'required|exists:saas_plans,id',
            'is_trial' => 'boolean',
        ]);

        $schoolId = auth()->user()->school_id;
        $isTrial = $request->get('is_trial', false);

        $subscription = $this->subscriptionService->createSubscription(
            $schoolId,
            $request->plan_id,
            $isTrial
        );

        return redirect()->route('kouffe-saas.school.dashboard')
            ->with('success', 'Abonnement créé avec succès');
    }

    /**
     * Annuler un abonnement
     */
    public function cancelSubscription(Request $request)
    {
        $request->validate([
            'subscription_id' => 'required|exists:saas_subscriptions,id',
            'reason' => 'nullable|string|max:500',
        ]);

        $this->subscriptionService->cancelSubscription(
            $request->subscription_id,
            $request->reason
        );

        return redirect()->back()
            ->with('success', 'Abonnement annulé avec succès');
    }

    /**
     * Configuration du domaine
     */
    public function domainSettings(Request $request)
    {
        $schoolId = auth()->user()->school_id;
        $domain = $this->domainService->getActiveDomain($schoolId);

        return view('kouffesaas::school-admin.domain-settings', compact('domain'));
    }

    /**
     * Configurer un domaine
     */
    public function setupDomain(Request $request)
    {
        $request->validate([
            'domain' => 'required|string|max:255',
            'type' => 'required|in:subdomain,custom',
        ]);

        $schoolId = auth()->user()->school_id;
        $result = $this->domainService->setupDomain($schoolId, $request->all());

        if ($result['success']) {
            return redirect()->route('kouffe-saas.school.domain-settings')
                ->with('success', $result['message']);
        }

        return redirect()->back()
            ->with('error', $result['message']);
    }

    /**
     * Vérifier un domaine
     */
    public function verifyDomain(Request $request)
    {
        $request->validate([
            'domain_id' => 'required|exists:saas_custom_domains,id',
        ]);

        $verified = $this->domainService->verifyDomain($request->domain_id);

        if ($verified) {
            return redirect()->back()
                ->with('success', 'Domaine vérifié avec succès');
        }

        return redirect()->back()
            ->with('error', 'Impossible de vérifier le domaine. Vérifiez vos paramètres DNS.');
    }

    /**
     * Liste des factures
     */
    public function invoices(Request $request)
    {
        $schoolId = auth()->user()->school_id;
        $invoices = SaasInvoice::where('school_id', $schoolId)
            ->with('subscription.plan')
            ->latest()
            ->paginate(20);

        return view('kouffesaas::school-admin.invoices', compact('invoices'));
    }

    /**
     * Calcule l'utilisation du stockage
     */
    protected function calculateStorageUsage(int $schoolId): float
    {
        // Logique pour calculer l'utilisation du stockage
        // À implémenter selon votre système de fichiers
        return 0.0;
    }
}

