<?php

namespace Modules\KouffeSaas\Services;

use Modules\KouffeSaas\Models\SaasCustomDomain;
use Modules\KouffeSaas\Models\SaasSubscription;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class DomainService
{
    protected $baseDomain;

    public function __construct()
    {
        $this->baseDomain = config('app.domain', 'kouffe.com');
    }

    /**
     * Configure un domaine (sous-domaine ou domaine personnalisé)
     */
    public function setupDomain(int $schoolId, array $data): array
    {
        $validator = Validator::make($data, [
            'domain' => 'required|string|max:255',
            'type' => 'required|in:subdomain,custom',
        ]);

        if ($validator->fails()) {
            return [
                'success' => false,
                'message' => $validator->errors()->first(),
            ];
        }

        // Vérifier que l'école a le droit au domaine personnalisé
        if ($data['type'] === 'custom') {
            $subscription = SaasSubscription::where('school_id', $schoolId)
                ->active()
                ->first();

            if (!$subscription || !$subscription->plan->custom_domain) {
                return [
                    'success' => false,
                    'message' => 'Votre plan ne permet pas d\'utiliser un domaine personnalisé.',
                ];
            }
        }

        // Vérifier si le domaine existe déjà
        $existingDomain = SaasCustomDomain::where('domain', $data['domain'])
            ->where('school_id', '!=', $schoolId)
            ->first();

        if ($existingDomain) {
            return [
                'success' => false,
                'message' => 'Ce domaine est déjà utilisé par une autre école.',
            ];
        }

        // Créer ou mettre à jour le domaine
        $domain = SaasCustomDomain::updateOrCreate(
            [
                'school_id' => $schoolId,
                'domain' => $data['domain'],
            ],
            [
                'type' => $data['type'],
                'subdomain' => $data['type'] === 'subdomain' ? $data['domain'] : null,
                'is_verified' => false,
                'is_active' => false,
                'verification_token' => Str::random(32),
            ]
        );

        return [
            'success' => true,
            'message' => 'Domaine configuré avec succès. Veuillez vérifier la propriété du domaine.',
            'domain' => $domain,
        ];
    }

    /**
     * Vérifie la propriété d'un domaine
     */
    public function verifyDomain(int $domainId): bool
    {
        $domain = SaasCustomDomain::findOrFail($domainId);
        
        // Logique de vérification DNS ou fichier
        // Pour l'instant, on simule la vérification
        $isVerified = $this->checkDomainOwnership($domain->domain, $domain->verification_token);

        if ($isVerified) {
            $domain->update([
                'is_verified' => true,
                'is_active' => true,
                'verified_at' => now(),
            ]);

            return true;
        }

        return false;
    }

    /**
     * Vérifie la propriété du domaine via DNS ou fichier
     */
    protected function checkDomainOwnership(string $domain, string $token): bool
    {
        // Vérification DNS TXT record
        $txtRecords = dns_get_record($domain, DNS_TXT);
        
        foreach ($txtRecords as $record) {
            if (isset($record['txt']) && str_contains($record['txt'], $token)) {
                return true;
            }
        }

        // Vérification via fichier HTML
        $verificationUrl = "http://{$domain}/.well-known/kouffe-verification.txt";
        $content = @file_get_contents($verificationUrl);
        
        if ($content && str_contains($content, $token)) {
            return true;
        }

        return false;
    }

    /**
     * Désactive un domaine
     */
    public function disableDomain(int $domainId): bool
    {
        $domain = SaasCustomDomain::findOrFail($domainId);
        $domain->update(['is_active' => false]);
        return true;
    }

    /**
     * Obtient le domaine actif pour une école
     */
    public function getActiveDomain(int $schoolId): ?SaasCustomDomain
    {
        return SaasCustomDomain::where('school_id', $schoolId)
            ->active()
            ->first();
    }
}

